/* BEGIN_COMMON_COPYRIGHT_HEADER
 * (c)LGPL2+
 *
 *
 * Copyright: 2012-2013 Boomaga team https://github.com/Boomaga
 * Authors:
 *   Alexander Sokoloff <sokoloff.a@gmail.com>
 *
 * This program or library is free software; you can redistribute it
 * and/or modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 *
 * END_COMMON_COPYRIGHT_HEADER */


#include "job.h"
#include "projectpage.h"

#include <QDebug>
#include <QIODevice>
#include <QFileInfo>
#include "pdfparser/pdfreader.h"
#include <QSharedDataPointer>

#define LOCK_CHECK          0
#define LOCK_CREATE         100
#define LOCK_DESTROY       -1

class JobData: public QSharedData
{
public:
    JobData();
    ~JobData();

    QString mFileName;
    qint64 mStartPos;
    qint64 mEndPos;
    QList<ProjectPage*> mPages;
    QString mTitle;
    QString mErrorString;

    static int lockUnlockFile(const QString &file, int lock);
    ProjectPage *takePage(ProjectPage *page);
};


/************************************************

 ************************************************/
JobData::JobData():
    mFileName(""),
    mStartPos(0),
    mEndPos(0)
{
}


/************************************************

 ************************************************/
JobData::~JobData()
{
    if (!mFileName.isEmpty())
    {
        lockUnlockFile(mFileName, LOCK_DESTROY);

        if (!lockUnlockFile(mFileName, LOCK_CHECK))
            QFile::remove(mFileName);
    }
    qDeleteAll(mPages);
}


/************************************************

 ************************************************/
int JobData::lockUnlockFile(const QString &file, int lock)
{
    if (file.isEmpty())
        return 0;

    int res = 0;
    static QHash<QString, int> items;
    if (items.contains(file))
        res = items[file];

    res += lock;
    items[file] = res;

    return res;
}


/************************************************

 ************************************************/
ProjectPage *JobData::takePage(ProjectPage *page)
{
    int n = mPages.indexOf(page);
    if (n>-1)
    {
        ProjectPage *page = mPages.takeAt(n);
        return page;
    }
    else
        return 0;
}


/************************************************

 ************************************************/
Job::Job():
    mData(new JobData())
{
}


/************************************************
 *
 ************************************************/
Job::Job(const Job &other):
    mData(other.mData)
{
}


/************************************************

 ************************************************/
Job::~Job()
{

}


/************************************************

 ************************************************/
Job &Job::operator =(const Job &other)
{
    mData = other.mData;
    return *this;
}


/************************************************
 *
 * ***********************************************/
bool Job::operator==(const Job &other) const
{
    return mData == other.mData;
}


/************************************************

 ************************************************/
int Job::pageCount() const
{
    return mData->mPages.count();
}


/************************************************

 ************************************************/
ProjectPage *Job::page(int index) const
{
    return mData->mPages[index];
}


/************************************************
 *
 * ***********************************************/
int Job::visiblePageCount() const
{
    int res =0;
    foreach (ProjectPage *p, mData->mPages)
    {
        if (p->visible())
            res++;
    }

    return res;
}


/************************************************

 ************************************************/
ProjectPage *Job::firstVisiblePage() const
{
    foreach (ProjectPage *page, mData->mPages)
    {
        if (page->visible())
            return page;
    }

    return 0;
}


/************************************************

 ************************************************/
int Job::indexOfPage(const ProjectPage *page, int from) const
{
    return mData->mPages.indexOf(const_cast<ProjectPage*>(page), from);
}


/************************************************
 *
 * ***********************************************/
void Job::insertPage(int before, ProjectPage *page)
{
    mData->mPages.insert(before, page);
}


/************************************************

 ************************************************/
void Job::addPage(ProjectPage *page)
{
    mData->mPages.append(page);
}


/************************************************
 *
 * ***********************************************/
void Job::removePage(ProjectPage *page)
{
    ProjectPage *p = mData->takePage(page);
    delete p;
}


/************************************************

 ************************************************/
void Job::removePages(const QList<ProjectPage*> &pages)
{
    foreach (ProjectPage *page, pages)
    {
        ProjectPage *p = mData->takePage(page);
        delete p;
    }
}


/************************************************

 ************************************************/
ProjectPage *Job::takePage(ProjectPage *page)
{
    int n = mData->mPages.indexOf(page);
    if (n>-1)
    {
        ProjectPage *page = mData->mPages.takeAt(n);
        return page;
    }
    else
        return 0;
}


/************************************************

 ************************************************/
QString Job::title(bool human) const
{
    if (mData->mTitle.isEmpty() && human)
        return QObject::tr("Untitled");

    return mData->mTitle;
}


/************************************************

 ************************************************/
void Job::setTitle(const QString &title)
{
    mData->mTitle = title;
}


/************************************************
 *
 ************************************************/
QString Job::fileName() const
{
    return mData->mFileName;
}


/************************************************
 *
 ************************************************/
void Job::setFileName(const QString &fileName)
{
    mData->mFileName = fileName;
}


/************************************************
 *
 ************************************************/
qint64 Job::fileStartPos() const
{
    return mData->mStartPos;
}


/************************************************
 *
 ************************************************/
qint64 Job::fileEndPos() const
{
    return mData->mEndPos;
}


/************************************************
 *
 ************************************************/
void Job::setFilePos(qint64 startPos, qint64 endPos)
{
    mData->mStartPos = startPos;
    mData->mEndPos   = endPos;
}


/************************************************

 ************************************************/
QString Job::errorString() const
{
    return mData->mErrorString;
}


/************************************************
 *
 * ***********************************************/
ProjectPage *Job::insertBlankPage(int before)
{
    ProjectPage *page = new ProjectPage();
    insertPage(before, page);
    return page;
}


/************************************************
 *
 ************************************************/
ProjectPage *Job::addBlankPage()
{
    ProjectPage *page = new ProjectPage();
    addPage(page);
    return page;
}


/************************************************
 *
 ************************************************/
Job Job::clone()
{
    Job res = *this;
    res.mData.detach();
    for (int i=0; i< mData->mPages.count(); ++i)
        res.mData->mPages[i] = mData->mPages.at(i)->clone();

    return res;
}


/************************************************
 *
 * ***********************************************/
JobList::JobList():
    QList<Job>()
{

}


/************************************************
 *
 * ***********************************************/
JobList::JobList(const QList<Job> &other):
    QList<Job>(other)
{
}


/************************************************

 ************************************************/
int JobList::indexOfProjectPage(const ProjectPage * page, int from) const
{
    if (!page)
        return -1;

    for(int i=from; i<this->count(); ++i)
    {
        if (at(i).indexOfPage(page) > -1)
            return i;
    }

    return -1;
}


