/*
 * ClearVideo decoder
 * Copyright (c) 2012-2018 Konstantin Shishkov
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef AVCODEC_CLEARVIDEODATA_H
#define AVCODEC_CLEARVIDEODATA_H

#include "libavutil/common.h"

#define MAX_VLC_ENTRIES 1370
#define NUM_DC_CODES 127
#define NUM_AC_CODES 104

static const uint8_t clv_dc_lens[NUM_DC_CODES] = {
    22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22,
    22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22,
    21, 21, 21, 20, 20, 20, 19, 19, 19, 19, 19, 19, 19, 19, 19, 18, 18, 18, 18,
    17, 17, 17, 17, 16, 16, 16, 15, 15, 15, 15, 14, 14, 14, 14, 14, 14, 14, 14,
    13, 13, 12, 12, 12, 12, 12, 12, 12, 12, 11, 11, 11, 11, 11, 11, 10, 10, 10,
    10, 10, 10, 10, 10,  9,  9,  9,  9,  9,  9,  9,  8,  8,  8,  8,  8,  7,  7,
     7,  7,  7,  7,  6,  6,  5,  5,  5,  5,  3,  3,  1,
};

static const uint8_t clv_dc_syms[NUM_DC_CODES] = {
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B,
    0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x14, 0x15, 0x6E, 0x6F, 0x70,
    0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7A, 0x7B, 0x7C,
    0x7D, 0x7E, 0x13, 0x68, 0x6D, 0x17, 0x18, 0x6A, 0x16, 0x19, 0x64, 0x65,
    0x66, 0x67, 0x69, 0x6B, 0x6C, 0x1A, 0x1B, 0x62, 0x63, 0x1D, 0x1E, 0x5F,
    0x61, 0x1F, 0x5E, 0x60, 0x1C, 0x21, 0x5C, 0x5D, 0x20, 0x24, 0x25, 0x26,
    0x58, 0x59, 0x5A, 0x5B, 0x23, 0x57, 0x22, 0x27, 0x28, 0x29, 0x53, 0x54,
    0x55, 0x56, 0x2A, 0x2B, 0x2C, 0x4F, 0x51, 0x52, 0x2D, 0x2E, 0x2F, 0x30,
    0x31, 0x32, 0x4E, 0x50, 0x33, 0x34, 0x35, 0x4A, 0x4B, 0x4C, 0x4D, 0x36,
    0x37, 0x47, 0x48, 0x49, 0x38, 0x39, 0x3A, 0x44, 0x45, 0x46, 0x3B, 0x43,
    0x3C, 0x3D, 0x41, 0x42, 0x3E, 0x40, 0x3F,
};

static const uint16_t clv_ac_syms[NUM_AC_CODES] = {
    0x0000, 0x1012, 0x1003, 0x000B, 0x000A, 0x11C1, 0x11B1, 0x11A1, 0x1191,
    0x0092, 0x0082, 0x0072, 0x0062, 0x0052, 0x0033, 0x0023, 0x0014, 0x000C,
    0x0015, 0x0171, 0x0181, 0x11D1, 0x11E1, 0x11F1, 0x1201, 0x0016, 0x0024,
    0x0043, 0x0053, 0x0063, 0x00A2, 0x0191, 0x01A1, 0x1211, 0x1221, 0x1231,
    0x1241, 0x1251, 0x1261, 0x1271, 0x1281, 0x1BFF, 0x0009, 0x0008, 0x1181,
    0x1171, 0x1161, 0x1151, 0x1141, 0x1131, 0x1121, 0x1111, 0x1002, 0x0161,
    0x0151, 0x0141, 0x0131, 0x0121, 0x0111, 0x0101, 0x00F1, 0x0042, 0x0032,
    0x0007, 0x0006, 0x1101, 0x10F1, 0x10E1, 0x10D1, 0x10C1, 0x10B1, 0x10A1,
    0x1091, 0x00E1, 0x00D1, 0x0022, 0x0013, 0x0005, 0x1081, 0x1071, 0x1061,
    0x1051, 0x00C1, 0x00B1, 0x00A1, 0x0004, 0x1041, 0x1031, 0x1021, 0x1011,
    0x0091, 0x0081, 0x0071, 0x0061, 0x0012, 0x0003, 0x0051, 0x0041, 0x0031,
    0x1001, 0x0001, 0x0011, 0x0021, 0x0002
};

/* The -9 refers to an open end in the tree. */
static const uint8_t clv_ac_bits[NUM_AC_CODES] = {
    -9, 11, 11, 11, 11, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 11, 11,
    11, 11, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12,  7, 10, 10,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  7,  7,  7,  7,  7,  7,  7,  7,  6,  6,  6,  6,  6,  6,  6,  6,  6,
     6,  5,  5,  5,  4,  2,  3,  4,  4
};

static const uint8_t clv_flags_bits[][16] = {
    { 2,  3,  4,  4,  4,  4,  4,  4,  4,  4,  5,  5,  5,  6,  7,  7 }, // Y_0
    { 2,  3,  3,  4,  4,  4,  4,  4,  4,  4,  6,  6,  6,  7,  8,  8 }, // Y_1
    { 1,  4,  4,  4,  4,  4,  4,  5,  5,  5,  6,  7,  8,  9, 10, 10 }, // Y_2
    { 1,  4,  4,  4,  4,  4,  4,  5,  5,  5,  7,  7,  7,  8,  9,  9 }, // U_0
    { 1,  4,  4,  4,  4,  4,  4,  4,  5,  6,  8,  8,  8,  9, 10, 10 }, // U_1
    { 1,  3,  4,  4,  4,  5,  5,  5,  5,  5,  6,  7,  8,  9, 10, 10 }, // V_0
    { 1,  3,  4,  4,  4,  4,  5,  5,  5,  6,  7,  8,  9, 10, 11, 11 }  // V_1
};

static const uint8_t clv_flags_syms[][16] = {
    { 15,  0,  3,  5, 12,  1, 10,  2,  4,  8,  7, 11, 14,  6,  9, 13 }, // Y_0
    {  0, 15,  3, 12,  5,  1,  4,  2,  8, 10, 11,  7,  9,  6, 13, 14 }, // Y_1
    {  0,  3, 12,  4,  2,  1,  8,  5,  7, 10,  9,  6, 11, 13, 14, 15 }, // Y_2
    {  0, 15,  3, 12,  1, 10,  2,  5,  4,  8, 11,  7, 14, 13,  9,  6 }, // U_0
    {  0,  2, 12,  3,  4,  1,  8,  5, 10,  7,  9,  6, 11, 14, 13, 15 }, // U_1
    {  0, 15,  1, 12,  3, 10,  2,  5,  8,  4, 11, 14,  6,  9,  7, 13 }, // V_0
    {  0,  8,  3,  4,  2,  1, 12,  5, 10,  7,  9,  6, 14, 11, 13, 15 }  // V_1
};

static const uint8_t clv_mv_len_counts[][16] = {
    {  1,  0,  0,  2,  2,  6, 13,  6, 14, 28,  38,  38,  58,  56,  20,   8 },
    {  1,  0,  0,  2,  3,  4,  8, 10, 20, 30,  37,  63,  84,  68,  32,   0 },
    {  0,  1,  0,  4,  0,  6,  9, 24, 32, 48, 112, 142, 232, 166, 142,  44 },
    {  0,  1,  0,  2,  2,  6,  9, 16, 40, 82, 146, 228, 191, 255, 244, 100 },
    {  0,  0,  2,  5,  2, 12,  6, 15, 14, 24,  37,  33,  24,  28,   8,  16 },
    {  1,  0,  0,  2,  2,  5, 10, 12, 19, 19,  37,  74,  69, 120,  48,  24 },
    {  0,  0,  2,  3,  8,  7,  8,  8, 23, 37,  30,  36,  28,  16,  12,   8 },
    {  1,  0,  0,  0,  4,  5, 14, 18, 20, 39,   6,  80,  75, 112,  36,  32 },
};

#define MV_ESC 0x13
static const uint16_t clv_mv_syms[] = {
    /* Y_0 motion vectors - 290 entries */
    0x0000, 0x00FF, 0x0001, 0xFF00, 0x0100, 0x0002, 0x00FE, MV_ESC, 0x00FD,
    0x0004, 0x00FC, 0x0003, 0x01FF, 0xFF01, 0x0101, 0xFFFF, 0xFE00, 0x0200,
    0x0005, 0x00FB, 0x00FA, 0x0006, 0x0007, 0x00F9, 0xFF02, 0x0102, 0xFFFE,
    0x01FE, 0xFD00, 0x0300, 0xFC00, 0x0400, 0x02FF, 0xFE01, 0x0201, 0xFEFF,
    0x0008, 0x00F8, 0xFB00, 0x0500, 0xFFFD, 0x01FD, 0xFF03, 0x0103, 0xFFFA,
    0x01FA, 0xFF06, 0x0106, 0xFA00, 0x0600, 0xFE02, 0x0202, 0xFEFE, 0x02FE,
    0xFFFC, 0x01FC, 0xFF04, 0x0104, 0xF900, 0x0700, 0xFF05, 0x0105, 0xFFFB,
    0x01FB, 0xFEFD, 0x02FD, 0xFE03, 0x0203, 0xFF07, 0x0107, 0xFFF9, 0x01F9,
    0xFEFC, 0x02FC, 0xFE04, 0x0204, 0xFD02, 0x0302, 0xFDFE, 0x03FE, 0x03FF,
    0xFD01, 0x0301, 0xFDFF, 0xFDFD, 0xFD03, 0x03FD, 0x0303, 0xFEFA, 0x02FA,
    0xFE06, 0x0206, 0xFFF8, 0x0108, 0x01F8, 0xFF08, 0x04FF, 0xFC01, 0x0401,
    0xFCFF, 0x0208, 0xFE08, 0x02F8, 0xFEF8, 0x02FB, 0xFE05, 0xFEFB, 0x0205,
    0xF800, 0x0800, 0xFD05, 0xFDFB, 0x0305, 0x03FB, 0xFDFC, 0x03FC, 0xFD04,
    0x0304, 0x0207, 0xFE07, 0xFEF9, 0x02F9, 0xFCFD, 0xFC03, 0x04FD, 0x0403,
    0xFC02, 0x0402, 0xFCFE, 0x04FE, 0xFB02, 0x05FE, 0x0502, 0xFBFE, 0xFCFC,
    0x04FC, 0xFC04, 0x0404, 0x0308, 0xFD08, 0x03F8, 0xFDF8, 0xFBFC, 0x05FC,
    0xFB04, 0x0504, 0x05FF, 0xFB01, 0x0501, 0xFBFF, 0x04FB, 0xFCFB, 0xFC05,
    0x0405, 0x06FF, 0xFA01, 0x0601, 0xFAFF, 0xFDFA, 0x03FA, 0xFD06, 0x0306,
    0xFBFD, 0xFB03, 0x05FD, 0x0503, 0xFCFA, 0x04FA, 0x0406, 0xFC06, 0xFC08,
    0x0408, 0xFCF8, 0x04F8, 0xFD07, 0x0307, 0xFDF9, 0x03F9, 0x05F8, 0x0508,
    0xFBF8, 0xFB08, 0x05FB, 0x0505, 0xFB05, 0xFBFB, 0x0407, 0xFC07, 0xFCF9,
    0x04F9, 0xFA03, 0xFAFD, 0x06FD, 0x0603, 0x06FE, 0xFAFE, 0xFA08, 0xFA02,
    0x0608, 0x0602, 0xFAF8, 0x06F8, 0x07FF, 0xF9FF, 0xF901, 0x0701, 0x08FF,
    0x0801, 0xF801, 0xF8FF, 0xF902, 0xF9FE, 0x0702, 0x07FE, 0x06FB, 0x0605,
    0xFA05, 0xFAFB, 0xFB07, 0xFBF9, 0x0507, 0x05F9, 0xF903, 0x07FD, 0xF9FD,
    0x0703, 0x06F9, 0x05FA, 0xFAF9, 0x0506, 0xFBFA, 0x0607, 0xFA07, 0xFB06,
    0x0606, 0xFAFA, 0xFA06, 0x06FA, 0xF9FA, 0x0706, 0xF906, 0x07FA, 0xF802,
    0xF8FE, 0x08FE, 0x0802, 0x07F8, 0xF9F8, 0xFAFC, 0x0708, 0xF908, 0x0604,
    0x06FC, 0xFA04, 0x0707, 0x07F9, 0xF907, 0xF9F9, 0xF9FB, 0x07FB, 0x0705,
    0xF905, 0x0803, 0xF8FD, 0xF803, 0x08FD, 0x0805, 0x08FB, 0xF8FB, 0xF807,
    0xF8F9, 0x0807, 0x08F9, 0xF805, 0xF904, 0x0704, 0xF9FC, 0x07FC, 0x0806,
    0x08FA, 0xF806, 0xF8FA, 0x0804, 0x08FC, 0xF804, 0xF8FC, 0x0808, 0xF8F8,
    0x08F8, 0xF808,
    /* Y_1 motion vectors - 362 entries */
    0x0000, 0x00FF, 0x0001, 0xFF00, 0x0100, MV_ESC, 0xFFFF, 0x01FF, 0xFF01,
    0x0101, 0x00FE, 0x0002, 0x0003, 0x00FD, 0x0004, 0x00FC, 0xFE00, 0x0200,
    0x0005, 0x00FB, 0x00FA, 0x0006, 0x0007, 0x00F9, 0xFFFE, 0x01FE, 0xFF02,
    0x0102, 0xFF03, 0x0103, 0xFFFD, 0x01FD, 0xFEFF, 0x02FF, 0xFE01, 0x0201,
    0x0008, 0x00F8, 0x00F7, 0x0009, 0xFD00, 0x0300, 0xFF04, 0x0104, 0xFFFC,
    0x01FC, 0xFC00, 0x0400, 0xFB00, 0x0500, 0xFA00, 0x0600, 0xFEFE, 0x02FE,
    0xFE02, 0x0202, 0xFFFA, 0x01FA, 0xFF06, 0x0106, 0xFF05, 0x0105, 0xFFFB,
    0x01FB, 0xFDFF, 0x03FF, 0xFD01, 0x0301, 0xFE03, 0x0203, 0xFEFD, 0x02FD,
    0xF900, 0x0700, 0xFF08, 0x0108, 0xFFF8, 0x01F8, 0xFCFF, 0x04FF, 0xFC01,
    0x0401, 0xFF07, 0x0107, 0xFFF9, 0x01F9, 0x0204, 0xFEFC, 0x02FC, 0xFE04,
    0xFE05, 0x0205, 0xFEFB, 0x02FB, 0x0109, 0xFFF7, 0xFF09, 0x01F7, 0xFDFE,
    0x03FE, 0xFD02, 0x0302, 0xFCFE, 0x04FE, 0xF800, 0x0800, 0xFC02, 0x0402,
    0xFD03, 0x0303, 0xFDFD, 0x03FD, 0xFBFF, 0xFB01, 0x0501, 0x05FF, 0xFE07,
    0x0207, 0xFEF9, 0x02F9, 0x04FD, 0xFC03, 0xFCFD, 0x0403, 0xFBFE, 0x05FE,
    0xFB02, 0x0502, 0xFEFA, 0x02FA, 0xFE06, 0x0206, 0x0304, 0xFDFC, 0x03FC,
    0xFD04, 0xFDFA, 0x03FA, 0x0306, 0x06FF, 0xFAFF, 0xFA01, 0x0601, 0xFD06,
    0xF700, 0x0900, 0x0404, 0xFCFC, 0x04FC, 0xFC04, 0xFD05, 0x0305, 0x03FB,
    0xFDFB, 0xFE08, 0xFEF8, 0x02F8, 0x0208, 0x03F8, 0xFD08, 0xFDF8, 0x0308,
    0x04F8, 0xFC08, 0x0408, 0xFCF8, 0xFAFE, 0x06FE, 0xFA02, 0x0602, 0x04FA,
    0xFC06, 0x0406, 0xFCFA, 0xFC05, 0x0405, 0x04FB, 0xFCFB, 0x05FD, 0xFB03,
    0xFBFD, 0x0503, 0x0504, 0xFBFC, 0xFB04, 0x05FC, 0x06FD, 0xFAFC, 0x0604,
    0x06FC, 0xFA03, 0xFAFD, 0x0603, 0xFA04, 0xFB05, 0x0505, 0xFD07, 0x0307,
    0xFBFB, 0xFDF9, 0x05FB, 0x03F9, 0x0508, 0xFBF8, 0x05F8, 0xFB08, 0x0209,
    0xFE09, 0x02F7, 0xFEF7, 0x0608, 0x06F8, 0xFA08, 0xFAF8, 0xFC07, 0xF9FF,
    0x0407, 0x07FF, 0xFCF9, 0xF901, 0x0701, 0x04F9, 0xF9FE, 0x07FE, 0xF902,
    0x0702, 0x05F9, 0xFB07, 0xFBF9, 0x0507, 0x07FD, 0xF903, 0xF9FD, 0x0703,
    0xFD09, 0x03F7, 0x0309, 0xFDF7, 0x0704, 0xF9FC, 0x07FC, 0xF904, 0x0605,
    0xFAFB, 0xFA05, 0x06FB, 0x05FA, 0xFB06, 0x0506, 0xFBFA, 0x08FF, 0xF801,
    0xF8FF, 0x0801, 0xF8FE, 0x0802, 0xF802, 0x08FE, 0x0901, 0xF7FF, 0x09FF,
    0xF701, 0x06F9, 0x0607, 0xFAF9, 0xFA07, 0x04F7, 0xFC09, 0x0409, 0xFCF7,
    0x0707, 0xF9F9, 0xF907, 0x07F9, 0x0606, 0xFA06, 0x06FA, 0xFAFA, 0xF908,
    0xF9F8, 0x07F8, 0x0708, 0x07FA, 0xF906, 0x0706, 0xF9FA, 0x0804, 0xF7FE,
    0x08FC, 0xF804, 0xF702, 0x0902, 0xF8FC, 0x09FE, 0x05F7, 0xFB09, 0x0509,
    0xFBF7, 0x0803, 0x08FD, 0xF803, 0xF8FD, 0xF905, 0x0705, 0x07FB, 0xF9FB,
    0xF8FA, 0x08FA, 0x0806, 0xF806, 0xF7FC, 0xF704, 0x09FC, 0x0904, 0xF7FD,
    0xF703, 0x09FD, 0x0903, 0x0808, 0xF808, 0x08F8, 0xF8F8, 0xF706, 0x0906,
    0xF7FA, 0x09FA, 0xF805, 0x0805, 0xF8FB, 0x08FB, 0xFA09, 0x0609, 0x06F7,
    0xFAF7, 0x08F9, 0xF8F9, 0x0807, 0xF807, 0x0709, 0x07F7, 0xF909, 0xF9F7,
    0x0905, 0xF705, 0x09FB, 0xF7FB, 0x09F8, 0xF7F8, 0x0908, 0xF708, 0x08F7,
    0xF809, 0xF8F7, 0x0809, 0xF709, 0xF7F7, 0x0909, 0x09F7, 0xF7F9, 0x09F9,
    0x0907, 0xF707,
    /* Y_2 motion vectors - 962 entries */
    0x0000, 0x0001, 0x00FF, 0xFF00, 0x0100, 0x0002, 0x00FE, 0x0101, 0xFFFF,
    0x01FF, 0xFF01, 0x0003, 0x00FD, 0x0004, 0x00FC, MV_ESC, 0x0005, 0x00FB,
    0xFE00, 0x0200, 0xFF02, 0x0102, 0xFFFE, 0x01FE, 0x00FA, 0x0006, 0x00F9,
    0x0007, 0x0201, 0xFEFF, 0x02FF, 0xFE01, 0xFD00, 0x0300, 0xFF03, 0x0103,
    0xFFFD, 0x01FD, 0xFC00, 0x0400, 0x0008, 0x00F8, 0xFB00, 0x0500, 0x0009,
    0x00F7, 0x00F5, 0x000B, 0xFE02, 0x0202, 0xFEFE, 0x02FE, 0x00F6, 0x000A,
    0xFF04, 0x0104, 0xFFFC, 0x01FC, 0xFA00, 0x0600, 0x0301, 0xFDFF, 0x03FF,
    0xFD01, 0xF900, 0x0700, 0x0401, 0xFCFF, 0x04FF, 0xFC01, 0x000C, 0x00F4,
    0xFFFA, 0x01FA, 0xFF06, 0x0106, 0xFF05, 0x0105, 0xFFFB, 0x01FB, 0xF800,
    0x0800, 0xFE03, 0x0203, 0xFEFD, 0x02FD, 0xFE04, 0x0204, 0xFEFC, 0x02FC,
    0xFFF9, 0x01F9, 0xFF07, 0x0107, 0x00F3, 0x000D, 0xFD02, 0x0302, 0xFDFE,
    0x03FE, 0x00F2, 0x000E, 0x0501, 0xFBFF, 0x05FF, 0xFB01, 0xFE05, 0x0205,
    0xFEFB, 0x02FB, 0xFF08, 0x0108, 0xFFF8, 0x01F8, 0xFC02, 0x0402, 0xFCFE,
    0x04FE, 0xF700, 0x0900, 0xFEFA, 0x02FA, 0xFE06, 0x0206, 0x00F1, 0x000F,
    0xFF09, 0x0109, 0xFFF7, 0x01F7, 0xFFF6, 0x01F6, 0xFF0A, 0x010A, 0xFD03,
    0x0303, 0xFDFD, 0x03FD, 0xFC03, 0x0403, 0xFCFD, 0x04FD, 0xFD04, 0x0304,
    0xFDFC, 0x03FC, 0xF600, 0x0A00, 0x0601, 0xFAFF, 0x06FF, 0xFA01, 0xFB02,
    0x0502, 0xFBFE, 0x05FE, 0xFF0C, 0x010C, 0xFFF4, 0x01F4, 0xFFF5, 0x01F5,
    0xFF0B, 0x010B, 0x0701, 0xF9FF, 0x07FF, 0xF901, 0xFDFA, 0x03FA, 0xFD06,
    0x0306, 0xFE08, 0x0208, 0xFEF8, 0x02F8, 0xFA02, 0x0602, 0xFAFE, 0x06FE,
    0xFEF9, 0x02F9, 0xFE07, 0x0207, 0xFE09, 0x0209, 0xFEF7, 0x02F7, 0xFC04,
    0x0404, 0xFCFC, 0x04FC, 0xFB03, 0x0503, 0xFBFD, 0x05FD, 0x0801, 0xF8FF,
    0x08FF, 0xF801, 0xFD05, 0x0305, 0xFDFB, 0x03FB, 0xFFF3, 0x01F3, 0x010D,
    0xFF0D, 0xF902, 0x0702, 0xF9FE, 0x07FE, 0xF500, 0x0B00, 0xFA03, 0x0603,
    0xFAFD, 0x06FD, 0xFC05, 0x0405, 0xFCFB, 0x04FB, 0xFEF6, 0x02F6, 0xFE0A,
    0x020A, 0xFDF9, 0x03F9, 0xFD07, 0x0307, 0xFEF5, 0x02F5, 0xFE0B, 0x020B,
    0xFFF2, 0x01F2, 0xFF0E, 0x010E, 0xF400, 0x0C00, 0xF802, 0x0802, 0xF8FE,
    0x08FE, 0xF903, 0x0703, 0xF9FD, 0x07FD, 0x03F8, 0xFB04, 0x0504, 0xFBFC,
    0x05FC, 0xFD08, 0x0308, 0xFDF8, 0xFB05, 0x0505, 0xFBFB, 0x05FB, 0xFA04,
    0x0604, 0xFAFC, 0x06FC, 0xF300, 0x0D00, 0xFC08, 0x0408, 0xFCF8, 0x04F8,
    0xFCFA, 0x04FA, 0xFC06, 0x0406, 0xF200, 0x0E00, 0x0901, 0xF7FF, 0x09FF,
    0xF701, 0xFD09, 0x0309, 0xFDF7, 0x03F7, 0xFDF6, 0x03F6, 0xFD0A, 0x030A,
    0xFC09, 0x0409, 0xFCF7, 0x04F7, 0x0A01, 0xF6FF, 0x0AFF, 0xF601, 0xFBF8,
    0xFB08, 0x0508, 0x05F8, 0x020C, 0xFE0C, 0xFEF4, 0x02F4, 0xFA05, 0xFAFB,
    0x0605, 0x06FB, 0x01F1, 0x010F, 0xFF0F, 0xFFF1, 0xF100, 0x0F00, 0xF904,
    0x0704, 0xF9FC, 0x07FC, 0xF803, 0x0803, 0xF8FD, 0x08FD, 0xF804, 0x0804,
    0xF8FC, 0x08FC, 0xFCF9, 0x04F9, 0xFC07, 0x0407, 0xF702, 0x0902, 0xF7FE,
    0x09FE, 0xF602, 0x0A02, 0xF6FE, 0x0AFE, 0xFB09, 0xFBF7, 0x05F7, 0x0509,
    0x0B01, 0xF5FF, 0x0BFF, 0xF501, 0xFBFA, 0x05FA, 0xFB06, 0x0506, 0xFBF9,
    0x05F9, 0xFB07, 0x0507, 0xF703, 0x0903, 0xF7FD, 0x09FD, 0xFAFA, 0xF704,
    0x06FA, 0x0904, 0xFA06, 0x0606, 0xF7FC, 0x09FC, 0xFDF5, 0x03F5, 0x030B,
    0xFD0B, 0xFAF9, 0x06F9, 0xFA07, 0x0607, 0xFA08, 0x0608, 0xFAF8, 0x06F8,
    0xF502, 0x0B02, 0xF5FE, 0x0BFE, 0xF9F9, 0x07F9, 0xF907, 0x0707, 0xF905,
    0xF9FB, 0x0705, 0x07FB, 0xF9FA, 0x07FA, 0xF906, 0x0706, 0x0C01, 0xF4FF,
    0x0CFF, 0xF401, 0xF908, 0x0708, 0xF9F8, 0x07F8, 0xF8F9, 0x08F9, 0xF807,
    0x0807, 0x0AFC, 0xF604, 0x0A04, 0xF6FC, 0xF8FA, 0x08FA, 0xF806, 0x0806,
    0xF603, 0x0A03, 0xF6FD, 0x0AFD, 0xF805, 0xF8FB, 0x0805, 0x08FB, 0xFA09,
    0x0609, 0xFAF7, 0x06F7, 0x040A, 0xFCF6, 0x04F6, 0xFC0A, 0xFDF4, 0x030C,
    0xFD0C, 0x03F4, 0xFB0A, 0xFBF6, 0x05F6, 0x050A, 0xFCF5, 0xFC0B, 0x04F5,
    0x040B, 0xF705, 0xF7FB, 0x0905, 0x09FB, 0xF808, 0x0808, 0xF8F8, 0x08F8,
    0xFAF6, 0x06F6, 0xFA0A, 0x060A, 0xF7FA, 0x09FA, 0xF706, 0x0906, 0xF9F6,
    0x07F6, 0xF90A, 0x070A, 0x02F1, 0xFEF3, 0x02F3, 0xFE0D, 0xFE0F, 0xFEF1,
    0x020F, 0x020D, 0x050B, 0xFBF5, 0x05F5, 0xFB0B, 0x040C, 0xFCF4, 0xFC0C,
    0x04F4, 0xFAF5, 0x06F5, 0xFA0B, 0x060B, 0x020E, 0xFEF2, 0xFE0E, 0x02F2,
    0xFB0C, 0x05F4, 0xFBF4, 0x050C, 0xF9F5, 0x070B, 0x07F5, 0xF90B, 0x030E,
    0xFDF2, 0x03F2, 0xFD0E, 0xF8F5, 0x08F5, 0x080B, 0xF80B, 0xF5FD, 0xF503,
    0x0B03, 0x0BFD, 0x030D, 0xFD0D, 0xFDF3, 0x03F3, 0x09F9, 0x0907, 0xF707,
    0xF7F9, 0xF6FA, 0x0AFA, 0x0A06, 0xF606, 0xF403, 0x0C03, 0x0709, 0xF4FD,
    0xF9F7, 0xF909, 0x0CFD, 0x07F7, 0x0AF9, 0x0A07, 0xF607, 0xF6F9, 0xF5FA,
    0xF506, 0x0BFA, 0x0B06, 0xF303, 0xF3FD, 0x0DFD, 0x0D03, 0x0D01, 0xF3FF,
    0x0DFF, 0xF301, 0xFCF2, 0xFC0E, 0x040E, 0x04F2, 0x0809, 0xF8F7, 0x08F7,
    0xF809, 0xF5FC, 0xF504, 0x0B04, 0x0BFC, 0xF7F5, 0x09F5, 0x090B, 0xF70B,
    0x0E03, 0xF2FD, 0xF203, 0x0EFD, 0xFBF2, 0x050E, 0x05F2, 0xFB0E, 0xFCF3,
    0xFC0D, 0x04F3, 0x040D, 0xF402, 0x0C02, 0xF4FE, 0x0CFE, 0xF6F5, 0xF6FB,
    0x0AF5, 0xF605, 0x0A05, 0x0AFB, 0x0A0B, 0xF60B, 0x080A, 0xF8F6, 0x08F6,
    0xF80A, 0xF7F6, 0xF70A, 0x09F6, 0x090A, 0x0BFB, 0xF505, 0xF5FB, 0x0B05,
    0x030F, 0x03F1, 0xFDF1, 0xFD0F, 0xF2FF, 0x0EFF, 0xF201, 0x0E01, 0xF6F6,
    0x0AF6, 0x0908, 0xF60A, 0x0A0A, 0xF708, 0xF7F8, 0x09F8, 0x0BF9, 0x0B07,
    0xF5F9, 0xF507, 0xF709, 0x0909, 0xF7F7, 0x09F7, 0x0CF9, 0xF407, 0x0C07,
    0xF4F9, 0xF608, 0x0A08, 0xF6F8, 0x0AF8, 0xFBF3, 0x05F3, 0xFB0D, 0x050D,
    0xF5F8, 0xF508, 0x0B08, 0x0BF8, 0x060D, 0xFAF3, 0x06F3, 0xFA0D, 0x0CFA,
    0xF4FA, 0xF406, 0x0C06, 0x0DF9, 0xF307, 0x0D07, 0xF3F9, 0x0D02, 0x0DFE,
    0xF3FE, 0xF302, 0xFA0E, 0xFAF2, 0x060E, 0x06F2, 0xF4FB, 0x0C05, 0x0CFB,
    0xF405, 0x070D, 0xF9F3, 0xF90D, 0x07F3, 0xF5F5, 0x0BF5, 0x0B0B, 0xF50B,
    0x0AF7, 0x0A09, 0xF6F7, 0xF609, 0xF202, 0x0E02, 0xFAF4, 0xF2FE, 0x0EFE,
    0xFA0C, 0x06F4, 0x060C, 0xF404, 0x0C04, 0xF4FC, 0x0CFC, 0xFC0F, 0x0D04,
    0xFCF1, 0x04F1, 0xF304, 0x0DFC, 0x040F, 0x070C, 0xF3FC, 0xF9F4, 0x07F4,
    0xF90C, 0xF8F3, 0xF3FA, 0x08F3, 0xF306, 0x080D, 0x0DFA, 0x0D06, 0xF80D,
    0xF204, 0x0EFC, 0xF2FC, 0x0E04, 0x07F2, 0xF90E, 0xF9F2, 0x070E, 0xF3FB,
    0x0D05, 0xF305, 0x0DFB, 0xF4F8, 0x080C, 0xF8F4, 0xF408, 0x0C08, 0x0CF8,
    0x08F4, 0xF80C, 0xF70C, 0x090C, 0xF7F4, 0x09F4, 0xF4F5, 0x0CF5, 0x0C0B,
    0xF40B, 0xF8F2, 0xF80E, 0x080E, 0x08F2, 0x0A0C, 0xF60C, 0xF6F4, 0x0AF4,
    0x0F02, 0xF102, 0xF1FE, 0x0FFE, 0x090D, 0xF7F3, 0x09F3, 0xF70D, 0xF1FC,
    0xF104, 0x0FFC, 0x0F04, 0x0B0C, 0xF50C, 0xF5F4, 0x0BF4, 0xFB0F, 0x050F,
    0x05F1, 0xFBF1, 0xF5F6, 0x0B0A, 0xF50A, 0x0BF6, 0xF60D, 0xF6F3, 0x0A0D,
    0x0AF3, 0xF70E, 0xF7F2, 0x09F2, 0x090E, 0x06F1, 0x060F, 0xFAF1, 0xFA0F,
    0x0DF5, 0x0D0B, 0xF3F5, 0xF30B, 0x0B09, 0x0BF7, 0xF509, 0xF5F7, 0x0D08,
    0xF3F8, 0xF308, 0x0DF8, 0x0CF6, 0x0C0A, 0xF4F6, 0xF40A, 0xF50D, 0x0BF3,
    0x0B0D, 0xF208, 0xF40C, 0xF5F3, 0x0EF8, 0xF4F4, 0xF2F8, 0x0CF4, 0x0C0C,
    0x0E08, 0xF2F9, 0x0EF9, 0xF207, 0x0E07, 0xF205, 0x0E05, 0x0EFB, 0xF2FB,
    0x0A0E, 0x0AF2, 0xF6F2, 0xF103, 0x0FFD, 0xF60E, 0x0F03, 0xF1FD, 0x0EF5,
    0x0E0B, 0xF2F5, 0xF20B, 0x0DF6, 0xF30A, 0xF3F6, 0x0D0A, 0x0F05, 0x0FFB,
    0xF105, 0xF1FB, 0x070F, 0xF9F1, 0x07F1, 0xF90F, 0xF80F, 0x080F, 0x08F1,
    0xF8F1, 0x0BF2, 0xF50E, 0xF5F2, 0x0B0E, 0x09F1, 0x090F, 0xF7F1, 0xF70F,
    0x0CF3, 0xF20A, 0xF40D, 0x0EF6, 0xF4F3, 0xF2F6, 0x0E0A, 0x0C0D, 0xF60F,
    0xF50F, 0xF4F7, 0x0A0F, 0x0AF1, 0x0BF1, 0xF6F1, 0x0C09, 0x0B0F, 0xF409,
    0x0CF7, 0xF5F1, 0xF40F, 0x0CF1, 0x0C0F, 0xF4F1, 0xF30C, 0x0DF4, 0xF3F4,
    0x0D0C, 0xF309, 0x0EF4, 0xF3F7, 0x0D09, 0x0DF7, 0xF2F4, 0xF20C, 0x0E0C,
    0xF4F2, 0xF40E, 0x0C0E, 0x0CF2, 0x0DF2, 0xF30E, 0xF3F2, 0x0D0E, 0x0EFA,
    0x0E06, 0xF206, 0xF2FA, 0x0FFA, 0x0F06, 0xF106, 0xF1FA, 0xF20E, 0x0E09,
    0x0E0E, 0xF2F2, 0xF2F7, 0x0EF2, 0x0EF7, 0xF209, 0xF2F3, 0xF30D, 0xF20D,
    0x0EF3, 0xF3F3, 0x0DF3, 0x0E0D, 0x0D0D, 0xF3F1, 0x0D0F, 0xF30F, 0x0DF1,
    0xF107, 0x0FF9, 0x0F07, 0xF1F9, 0xF2F1, 0xF20F, 0x0FF7, 0x0EF1, 0xF1F7,
    0x0E0F, 0xF109, 0x0F09, 0xF1F8, 0xF108, 0x0FF8, 0x0F08, 0x0F0B, 0xF10B,
    0x0FF6, 0xF1F6, 0x0FF5, 0x0F0A, 0xF10A, 0xF1F5, 0xF1F3, 0xF10D, 0x0FF3,
    0x0F0D, 0x0FFF, 0xF1FF, 0xF101, 0x0F01, 0x0F0F, 0xF1F1, 0x0FF1, 0xF10F,
    0x0FF2, 0xF10C, 0x0F0E, 0x0FF4, 0xF1F4, 0xF1F2, 0x0F0C, 0xF10E,
    /* Y_3 motion vectors - 1370 elements */
    0x0000, 0x00FF, 0x0001, 0xFF00, 0x0100, 0x00FE, 0x0002, 0xFFFF, 0x01FF,
    0xFF01, 0x0101, 0x00FD, 0x0003, 0x00FC, 0x0004, 0xFE00, 0x0200, 0x0005,
    0x00FB, MV_ESC, 0xFFFE, 0x01FE, 0xFF02, 0x0102, 0x0006, 0x00FA, 0xFEFF,
    0x02FF, 0xFE01, 0x0201, 0xFD00, 0x0300, 0xFC00, 0x0400, 0x00F9, 0x0007,
    0xFFFD, 0x01FD, 0xFF03, 0x0103, 0x00F8, 0x0008, 0x00F7, 0x0009, 0x000A,
    0x00F6, 0xFEFE, 0x02FE, 0xFE02, 0x0202, 0xFFFC, 0x01FC, 0xFF04, 0x0104,
    0xFB00, 0x0500, 0xFDFF, 0x03FF, 0xFD01, 0x0301, 0xFEFD, 0x02FD, 0xFE03,
    0x0203, 0xFCFF, 0x04FF, 0xFC01, 0x0401, 0xFF05, 0x0105, 0xFFFB, 0x01FB,
    0x000B, 0x00F5, 0xFA00, 0x0600, 0xF900, 0x0700, 0xFFF9, 0x01F9, 0xFF07,
    0x0107, 0xFDFE, 0x03FE, 0xFD02, 0x0302, 0xFBFF, 0x05FF, 0xFB01, 0x0501,
    0xF800, 0x0800, 0xFF06, 0x0106, 0xFFFA, 0x01FA, 0xFAFF, 0x06FF, 0xFA01,
    0x0601, 0xFEFC, 0x02FC, 0xFE04, 0x0204, 0xFFF8, 0x01F8, 0xFF08, 0x0108,
    0xFCFE, 0x04FE, 0xFC02, 0x0402, 0xF700, 0x0900, 0xF600, 0x0A00, 0xFDFD,
    0x03FD, 0xFD03, 0x0303, 0xFF0A, 0x010A, 0xFFF6, 0x01F6, 0xFE05, 0x0205,
    0xFEFB, 0x02FB, 0x000C, 0x00F4, 0xFDFC, 0x03FC, 0xFD04, 0x0304, 0x000D,
    0x00F3, 0xFE06, 0x0206, 0xFEFA, 0x02FA, 0xFFF7, 0x01F7, 0xFF09, 0x0109,
    0xFF0B, 0x010B, 0xFFF5, 0x01F5, 0xFBFE, 0x05FE, 0xFB02, 0x0502, 0xFD05,
    0x0305, 0xFDFB, 0x03FB, 0x000E, 0x00F2, 0xFCFD, 0x04FD, 0xFC03, 0x0403,
    0xF9FF, 0x07FF, 0xF901, 0x0701, 0xFBFD, 0x05FD, 0xFB03, 0x0503, 0xFEF9,
    0x02F9, 0xFE07, 0x0207, 0xF8FF, 0x08FF, 0xF801, 0x0801, 0x00F1, 0x000F,
    0xFAFE, 0x06FE, 0xFA02, 0x0602, 0xFD06, 0x0306, 0xFDFA, 0x03FA, 0x0010,
    0x00F0, 0xFEF8, 0x02F8, 0xFE08, 0x0208, 0xFE0A, 0x020A, 0xFEF6, 0x02F6,
    0xFEF7, 0x02F7, 0xFE09, 0x0209, 0xFDF9, 0x03F9, 0xFD07, 0x0307, 0xF500,
    0x0B00, 0xFCFC, 0x04FC, 0xFC04, 0x0404, 0xF7FF, 0x09FF, 0xF701, 0x0901,
    0xF9FE, 0x07FE, 0xF902, 0x0702, 0x0011, 0x00EF, 0xFBFC, 0x05FC, 0xFB04,
    0x0504, 0xFC05, 0x0405, 0xFCFB, 0x04FB, 0xFAFD, 0x06FD, 0xF8FE, 0x08FE,
    0xF802, 0x0802, 0xFA03, 0x0603, 0xFDF7, 0x03F7, 0xFD09, 0x0309, 0xFB05,
    0x0505, 0xFBFB, 0x05FB, 0xF9FD, 0x07FD, 0xF903, 0x0703, 0x00EE, 0x0012,
    0xFC06, 0x0406, 0xFCFA, 0x04FA, 0xF7FE, 0x09FE, 0xF702, 0x0902, 0xFDF8,
    0x03F8, 0xFD08, 0x0308, 0xF6FF, 0x0AFF, 0xF601, 0x0A01, 0xFAFC, 0x06FC,
    0xFA04, 0x0604, 0xFA05, 0x0605, 0xFAFB, 0x06FB, 0xFF0C, 0x010C, 0xFFF4,
    0x01F4, 0xFCF8, 0x04F8, 0xFC08, 0x0408, 0xFB06, 0x0506, 0xFBFA, 0x05FA,
    0xFE0B, 0x020B, 0xFEF5, 0x02F5, 0xFCF7, 0x04F7, 0xFC09, 0x0409, 0xFD0A,
    0x030A, 0xFDF6, 0x03F6, 0xFCF9, 0x04F9, 0xFC07, 0x0407, 0xF8FD, 0x08FD,
    0xF803, 0x0803, 0xF9FC, 0x07FC, 0xF904, 0x0704, 0xF400, 0x0C00, 0xF7FD,
    0x09FD, 0xF703, 0x0903, 0xF8FC, 0x08FC, 0xF804, 0x0804, 0xFBF9, 0x05F9,
    0xFB07, 0x0507, 0xF905, 0x0705, 0xF9FB, 0x07FB, 0xFAF9, 0x06F9, 0xFA07,
    0x0607, 0xFBF8, 0x05F8, 0xFB08, 0x0508, 0xFD0B, 0x030B, 0xFDF5, 0x03F5,
    0xFA06, 0x0606, 0xFAFA, 0x06FA, 0xFC0A, 0x040A, 0xFCF6, 0x04F6, 0xFAF8,
    0x06F8, 0xFA08, 0x0608, 0xF9F9, 0x07F9, 0xF907, 0x0707, 0xFF0E, 0x010E,
    0xFFF2, 0x01F2, 0xFBF7, 0x05F7, 0xFB09, 0x0509, 0xFF0D, 0x010D, 0xFFF3,
    0x01F3, 0xF906, 0xF9FA, 0x0706, 0x07FA, 0x0609, 0xFAF7, 0x06F7, 0xFA09,
    0x0709, 0xF9F7, 0x07F7, 0xF909, 0xF8F9, 0x08F9, 0xF807, 0x0807, 0xF300,
    0x0D00, 0xFB0A, 0x050A, 0xFBF6, 0x05F6, 0xF6FE, 0x0AFE, 0xF602, 0x0A02,
    0xFC0B, 0xFCF5, 0x040B, 0x04F5, 0xF7FC, 0x09FC, 0xF704, 0x0904, 0xF805,
    0x0805, 0xF8FB, 0x08FB, 0xF6FC, 0x0AFC, 0xF604, 0x0A04, 0xFE0D, 0x020D,
    0xFEF3, 0x02F3, 0x0809, 0xF8F7, 0x08F7, 0xF809, 0xF806, 0xF8FA, 0x0806,
    0x08FA, 0x0AFD, 0xF603, 0x0A03, 0xF6FD, 0xFA0A, 0x060A, 0xFAF6, 0x06F6,
    0xF5FF, 0x0BFF, 0xF501, 0x0B01, 0xF705, 0x0905, 0xF7FB, 0x09FB, 0xFFF1,
    0x01F1, 0xFF0F, 0x010F, 0xF605, 0x0A05, 0xF6FB, 0x0AFB, 0xFF11, 0x0111,
    0xFFEF, 0x01EF, 0xF9F8, 0x07F8, 0xF908, 0x0708, 0xF90A, 0x070A, 0xF9F6,
    0x07F6, 0xFB0B, 0x05F5, 0xFBF5, 0x050B, 0x080A, 0xF80A, 0xF8F6, 0x08F6,
    0xF7F9, 0x09F9, 0xF706, 0xF7FA, 0x0906, 0xF707, 0x09FA, 0x0907, 0xF606,
    0xF6FA, 0x0A06, 0x0AFA, 0xF6F9, 0x0AF9, 0xF607, 0x0A07, 0xF200, 0x0E00,
    0xF70A, 0x090A, 0xF5FD, 0x0BFD, 0xF7F6, 0x09F6, 0xF503, 0x0B03, 0xF60A,
    0x0A0A, 0xF6F6, 0x0AF6, 0x0110, 0xFFF0, 0x01F0, 0xFF10, 0xF8F8, 0x08F8,
    0xF808, 0x0808, 0xF7F8, 0x09F8, 0xF708, 0x0908, 0xF100, 0x0F00, 0xFA0B,
    0xFAF5, 0x060B, 0x06F5, 0xF9F5, 0xF90B, 0x07F5, 0x070B, 0xF4FF, 0x0CFF,
    0xF401, 0x0C01, 0xFFEE, 0x0112, 0x01EE, 0xFF12, 0x0909, 0xF7F7, 0x09F7,
    0xF709, 0xF505, 0x0B05, 0xF5FB, 0x0BFB, 0xF4FD, 0x0CFD, 0xF403, 0x0C03,
    0xFE0C, 0x020C, 0xFEF4, 0x02F4, 0x0CFB, 0x0A09, 0xF6F7, 0x0AF7, 0xF405,
    0x0C05, 0xF4FB, 0xF609, 0xF8F5, 0x080B, 0x08F5, 0xF80B, 0xF5FE, 0x0BFE,
    0xF502, 0x0B02, 0xFD0D, 0x030D, 0xFDF3, 0x03F3, 0x020E, 0xFEF2, 0x02F2,
    0xFE0E, 0x0B0A, 0xF5F6, 0x0BF6, 0xF50A, 0x0A08, 0xF6F8, 0x0AF8, 0xF608,
    0xF5FC, 0xF504, 0x0B04, 0x0BFC, 0x030C, 0xFD0C, 0xFDF4, 0x03F4, 0xFEEE,
    0xFE12, 0x02EE, 0x0212, 0xF7F5, 0xF70B, 0x090B, 0x09F5, 0xF4FE, 0x0CFE,
    0xF402, 0x0C02, 0xF40A, 0xF4FC, 0x0C0A, 0xF4F6, 0x0CF6, 0xF404, 0x0C04,
    0x0CFC, 0x02F1, 0xFEF1, 0x020F, 0xFE0F, 0x040D, 0xFC0D, 0xFCF3, 0x04F3,
    0x0B09, 0xFE11, 0x0211, 0xF5F7, 0x0BF7, 0xFEEF, 0x02EF, 0xF509, 0xF506,
    0xF5FA, 0x0B06, 0x0BFA, 0xFC0C, 0x040C, 0xFCF4, 0x04F4, 0xF406, 0xF4FA,
    0x0C06, 0x0CFA, 0xF3FD, 0x0DFD, 0xF303, 0x0D03, 0xF000, 0x1000, 0xFB0D,
    0xFBF3, 0x05F3, 0x050D, 0xF5F9, 0x0BF9, 0xF507, 0x0B07, 0xF60B, 0xF6F5,
    0x0A0B, 0x0AF5, 0xF4F9, 0x0CF9, 0xF407, 0x0C07, 0xF5F5, 0xF50B, 0x0B0B,
    0x0BF5, 0xEF00, 0x1100, 0x060D, 0xFAF3, 0x06F3, 0xFA0D, 0x050C, 0xF3FE,
    0xFB0C, 0x0DFE, 0x05F4, 0xF302, 0x0D02, 0xFBF4, 0x03F1, 0x030F, 0xFD0F,
    0xFDF1, 0xEE00, 0x1200, 0xFA0C, 0x06F4, 0x060C, 0xFAF4, 0xF2FE, 0x0EFE,
    0xF3FF, 0x0DFF, 0xF301, 0x0D01, 0xF202, 0x0E02, 0xF5F8, 0x0BF8, 0xF508,
    0x0B08, 0xF90D, 0x070D, 0xF9F3, 0x07F3, 0x030E, 0x03F2, 0xFDF2, 0xFD0E,
    0xF304, 0x0D04, 0x0DFC, 0xF3FC, 0x0DFB, 0xF3FB, 0xF305, 0x0D05, 0x070C,
    0xF90C, 0xF9F4, 0x07F4, 0xFDEE, 0xF2FF, 0x0EFF, 0x0312, 0xFD12, 0xF201,
    0x03EE, 0x0E01, 0xF80C, 0x080C, 0x08F4, 0xF8F4, 0x0DFA, 0xFC0E, 0xFCF2,
    0x040E, 0x04F2, 0xF306, 0xF3FA, 0x0D06, 0xF4F8, 0x0CF8, 0xF408, 0x0C08,
    0x0C09, 0xF4F7, 0x0CF7, 0xF409, 0x04F1, 0xFC0F, 0x040F, 0xFCF1, 0x0EFD,
    0xF2FD, 0xF203, 0x0E03, 0x090C, 0xF70C, 0xF7F4, 0x09F4, 0xF3F8, 0x0DF8,
    0xF308, 0x0D08, 0x0A0C, 0xF60C, 0xF6F4, 0x0AF4, 0xF4F5, 0x0C0B, 0xF40B,
    0x0CF5, 0x05F1, 0xFB0F, 0xF1FF, 0x0FFF, 0xF101, 0x0F01, 0xFBF1, 0x050F,
    0xFB0E, 0x050E, 0xFBF2, 0x05F2, 0x06F1, 0xFA0F, 0x060F, 0xFAF1, 0x0DF9,
    0xF3F9, 0x0D07, 0xF307, 0x080D, 0xF80D, 0xF8F3, 0x08F3, 0x090D, 0xF70D,
    0xF7F3, 0x09F3, 0xF3F7, 0xFE10, 0xFEF0, 0x0DF7, 0x02F0, 0x0210, 0x0D09,
    0xF309, 0xF0FF, 0x10FF, 0xF001, 0x1001, 0x0A0D, 0xFA0E, 0xF60D, 0x060E,
    0xFAF2, 0x06F2, 0xF6F3, 0x0AF3, 0xF2FC, 0x0EFC, 0x0E04, 0xF204, 0xF205,
    0x0EFB, 0x0E05, 0xF2FB, 0xF90F, 0x070F, 0x07F1, 0xF9F1, 0x0310, 0xFDF0,
    0xFD10, 0x03F0, 0x0E08, 0xF2F8, 0x0EF8, 0xF208, 0xF30A, 0xF3F6, 0x0DF6,
    0x0D0A, 0x0F08, 0xF1F8, 0x0FF8, 0xF108, 0xF50C, 0x0B0C, 0x0FFD, 0x0F03,
    0x0BF4, 0xF103, 0xF5F4, 0xF1FD, 0xF20A, 0xF2F6, 0x0EF6, 0x0410, 0xFCF0,
    0xFC10, 0x04F0, 0x0E0A, 0xF207, 0x0311, 0xFC11, 0x0411, 0xF2F9, 0xFCEF,
    0xFD11, 0xFDEF, 0x0EF9, 0x03EF, 0x0E07, 0x04EF, 0x0510, 0xFBF0, 0xFB10,
    0x0511, 0x05F0, 0xFBEF, 0xFB11, 0x05EF, 0x0F07, 0xF1F9, 0x0FF9, 0xF107,
    0xFA10, 0x0610, 0xFAF0, 0x06F0, 0xF0FD, 0xF003, 0x10FD, 0x1003, 0xF1FE,
    0x0FFE, 0xF102, 0x0F02, 0xF9F2, 0x07F2, 0xF90E, 0x070E, 0xF30B, 0x0D0B,
    0xF80E, 0xF3F5, 0x08F2, 0x080E, 0x0DF5, 0xF8F2, 0xF50D, 0xF5F3, 0x0B0D,
    0x0EFA, 0x0BF3, 0xF206, 0xF2FA, 0x0E06, 0xF70E, 0x090E, 0xF7F2, 0x09F2,
    0x0FFB, 0x0FFC, 0xF1FB, 0xF104, 0x0F04, 0xF105, 0xF1FC, 0x0F05, 0x0F0A,
    0x0FF6, 0x0F06, 0xF1F6, 0xF106, 0xF1FA, 0xF10A, 0x0FFA, 0xEFFD, 0xEF01,
    0x11FD, 0xEFFF, 0x11FF, 0xEF03, 0x1101, 0x1103, 0x0C0C, 0xF40C, 0xF4F4,
    0x0CF4, 0x10FB, 0xF005, 0xF0FB, 0x1005, 0x1002, 0xF0FE, 0x10FE, 0xF002,
    0xF910, 0x08F0, 0xF8F0, 0xF9F0, 0x0810, 0xF810, 0x07F0, 0x0710, 0x10FC,
    0xF0FC, 0xF60E, 0xF6F2, 0x1004, 0x0AF2, 0xF004, 0x0A0E, 0x11FC, 0xEFFC,
    0xF30C, 0xEF04, 0x1104, 0xF3F4, 0x0D0C, 0x0DF4, 0x11FB, 0x1105, 0xEFFB,
    0xEF05, 0x080F, 0xF80F, 0x08F1, 0xF8F1, 0xFC12, 0x04EE, 0x0412, 0xFCEE,
    0xEFFE, 0xF5F2, 0x11FE, 0x0B0E, 0x10FA, 0xF50E, 0x0BF2, 0xEF02, 0x1006,
    0x1102, 0xF006, 0xF0FA, 0xFB12, 0xFBEE, 0x0512, 0x05EE, 0x10F8, 0xF4F2,
    0x1008, 0x0CF2, 0xF0F8, 0xF40E, 0x0C0E, 0xF008, 0xF20C, 0x0EF4, 0xF2F4,
    0x0E0C, 0xF20B, 0xF2F5, 0x09F0, 0xF7F0, 0x0910, 0xF710, 0x0E0B, 0x0EF5,
    0x0AF0, 0x0A10, 0xF6F0, 0xF610, 0xEEFC, 0xFAEF, 0x12FC, 0x1204, 0xEE04,
    0x0611, 0xFA11, 0x06EF, 0xEF06, 0x1106, 0xEFFA, 0x11FA, 0xEF08, 0xEFF8,
    0x11F8, 0x1108, 0xF7F1, 0x090F, 0xF70F, 0x09F1, 0x0E09, 0xF209, 0x0EF7,
    0xF2F7, 0xF0F6, 0x100A, 0x10F6, 0xF5F0, 0xF00A, 0x0BF0, 0xF510, 0x0B10,
    0x0D0E, 0xF30E, 0x0A0F, 0x0DF2, 0xF3F2, 0xF60F, 0x0AF1, 0xF6F1, 0x0CF3,
    0x0FF7, 0x0F09, 0xF40D, 0xF4F3, 0xF1F7, 0x0C0D, 0xF109, 0x07EE, 0xF912,
    0xFA12, 0xFAEE, 0x0712, 0xF9EE, 0x0612, 0x06EE, 0xF9EF, 0xF911, 0x0711,
    0x07EF, 0x0D0D, 0x0DF3, 0x0F0C, 0x110A, 0x11F6, 0xF10C, 0xF30D, 0xEFF6,
    0x0FF4, 0xEF0A, 0xF3F3, 0xF1F4, 0x10F9, 0xF007, 0xF0F9, 0x1007, 0xF8EF,
    0x0811, 0xF20D, 0x0EF3, 0x0E0D, 0xF2F3, 0xF811, 0x08EF, 0xF1F5, 0x0FF5,
    0xF10B, 0x0F0B, 0xF0F4, 0xF00C, 0xF2F2, 0x10F4, 0x0E0E, 0x0EF2, 0xF20E,
    0x100C, 0x09EF, 0x0911, 0xF7EF, 0xF711, 0xF611, 0x0A11, 0xF6EF, 0x0AEF,
    0x12FF, 0xEE01, 0x1201, 0xEEFF, 0xF5F1, 0x0BF1, 0xF50F, 0x0B0F, 0x10F5,
    0xEE06, 0xF00B, 0x1206, 0x12FA, 0xF0F5, 0xEEFA, 0x100B, 0x0CF0, 0xF410,
    0xF4F0, 0x0C10, 0x11F9, 0x0BEF, 0xF5EF, 0x0CF1, 0xEFF9, 0xF511, 0x0C0F,
    0xEF07, 0xF4F1, 0xF40F, 0x0B11, 0x1107, 0xEEFD, 0x12FD, 0xEE03, 0x1203,
    0x0DF0, 0xF3F0, 0xF310, 0x0D10, 0x0C11, 0x0CEF, 0xF4EF, 0xF411, 0x0EF0,
    0xF210, 0x0E10, 0xF2F0, 0x08EE, 0xF8EE, 0x0812, 0xF812, 0x110B, 0xF7EE,
    0xF712, 0x11F5, 0x0912, 0xEFF5, 0x09EE, 0xEF0B, 0xEEF9, 0x12F9, 0xEE07,
    0x1207, 0x0F0E, 0xF30F, 0x0D0F, 0x0DF1, 0xF10E, 0xF3F1, 0xF1F2, 0x0FF2,
    0xF6EE, 0xF612, 0x0AEE, 0x0A12, 0xF20F, 0x0E0F, 0xF2F1, 0x0EF1, 0xF0F7,
    0x1009, 0xF009, 0x10F7, 0xEEFB, 0x12FB, 0xEE05, 0x1205, 0x1202, 0x0FF3,
    0xEE08, 0xF1F3, 0x12F8, 0x12FE, 0xF10D, 0x1208, 0x0F0D, 0xEEFE, 0xEEF8,
    0xEE02, 0x0FF1, 0xF10F, 0xF1F1, 0x0F0F, 0x12F4, 0x11F4, 0x1109, 0xEFF4,
    0xEE0A, 0xEFF7, 0x110C, 0xEEF4, 0xEE0C, 0x11F7, 0xEF0C, 0x120C, 0xF110,
    0xF1F0, 0x120A, 0xEEF6, 0xEF09, 0x12F6, 0x0F10, 0x0FF0, 0x10F2, 0xF0F2,
    0x100E, 0xF00E, 0xEEF5, 0x0B12, 0x0BEE, 0xF512, 0x120B, 0x12F5, 0xF5EE,
    0xEE0B, 0x1209, 0x12F7, 0xEE09, 0xEEF7, 0xF4EE, 0x0C12, 0x0CEE, 0xF412,
    0x100D, 0x0D12, 0xF0F3, 0x0DEE, 0xF3EF, 0x0DEF, 0xF3EE, 0x0D11, 0xF312,
    0xF00D, 0xF311, 0x10F3, 0x11F3, 0xEFF3, 0x0E12, 0xF212, 0xF2EE, 0x0EEE,
    0xEF0D, 0x110D, 0x100F, 0xF0F1, 0xF00F, 0x10F1, 0x10F0, 0xF211, 0xF0F0,
    0xF010, 0x0EEF, 0xF2EF, 0x0E11, 0x1010, 0x120D, 0xEE0D, 0x12F3, 0xEEF3,
    0x11F1, 0xF111, 0xEFF2, 0x110F, 0xEF0E, 0xEF0F, 0x0FEF, 0x11F2, 0x0F11,
    0xF1EF, 0x110E, 0xEFF1, 0xEE0E, 0x120E, 0x12F2, 0xF1EE, 0x0FEE, 0xEEF2,
    0xF112, 0x0F12, 0xF0EE, 0x10EE, 0xF012, 0x1012, 0x11F0, 0xEF10, 0xEFF0,
    0x1110, 0xF0EF, 0x10EF, 0x1011, 0xF011, 0x11EF, 0xEF11, 0xEFEF, 0xEF12,
    0x11EE, 0x1112, 0xEFEE, 0x1111, 0xEE11, 0xEEEF, 0x12EF, 0x1211, 0x1212,
    0x12EE, 0xEEEE, 0xEE12, 0xEEF1, 0x12F1, 0xEE0F, 0x120F, 0xEE10, 0x1210,
    0xEEF0, 0x12F0,
    /* U_1 motion vectors - 226 entries */
    0x0001, 0x00FF, 0x0000, 0x0002, 0x00FE, 0x00FD, 0x0003, 0xFF00, 0x0100,
    0xFE00, 0x0200, 0x0101, 0xFFFF, 0x01FF, 0xFF01, 0x00FB, 0x0005, 0xFD00,
    0x0300, 0xFC00, 0x0400, 0x0004, 0x00FC, MV_ESC, 0x00FA, 0x0006, 0x0201,
    0xFEFF, 0x02FF, 0xFE01, 0xFFFD, 0x01FD, 0xFF03, 0x0103, 0xFEFD, 0x02FD,
    0xFE03, 0x0203, 0xFF02, 0x0102, 0xFFFE, 0x01FE, 0xFB00, 0x0500, 0xFDFD,
    0x03FD, 0xFD03, 0x0303, 0xFF04, 0x0104, 0xFFFC, 0x01FC, 0xFE02, 0x0202,
    0xFEFE, 0x02FE, 0xFA00, 0x0600, 0xFFFB, 0x01FB, 0xFF05, 0x0105, 0x00F9,
    0x0007, 0xFD02, 0x0302, 0xFDFE, 0x03FE, 0x0301, 0xFDFF, 0x03FF, 0xFD01,
    0xFEFB, 0x02FB, 0xFE05, 0x0205, 0x0401, 0xFCFF, 0x04FF, 0xFC01, 0xFE04,
    0x0204, 0xFEFC, 0x02FC, 0xFC02, 0x0402, 0xFCFE, 0x04FE, 0xFD04, 0x0304,
    0xFDFC, 0x03FC, 0xFFFA, 0x01FA, 0x0106, 0xFF06, 0xFC04, 0x0404, 0xFCFC,
    0x04FC, 0x0501, 0xFBFF, 0x05FF, 0xFB01, 0x0504, 0xFB04, 0xFBFC, 0x05FC,
    0xFE06, 0xFEFA, 0x02FA, 0x0206, 0xF900, 0x0700, 0xFB02, 0x0502, 0xFBFE,
    0x05FE, 0x0306, 0xFDFA, 0xFD06, 0x03FA, 0xFCFA, 0x0406, 0xFC06, 0x04FA,
    0x01F9, 0x0107, 0xFF07, 0xFFF9, 0xFE07, 0x02F9, 0xFEF9, 0x0207, 0xFDFB,
    0x03FB, 0xFD05, 0x0305, 0xFCFD, 0x04FD, 0xFC03, 0x0403, 0x0601, 0x06FF,
    0xFAFF, 0xFA01, 0x0701, 0xF9FF, 0x07FF, 0xF901, 0x03F9, 0xFD07, 0x0307,
    0xFDF9, 0xFBFA, 0x0506, 0xFB06, 0x05FA, 0xFCFB, 0x04FB, 0x0405, 0xFC05,
    0x0503, 0x05FD, 0xFB03, 0xFBFD, 0xFA03, 0xFAFA, 0x0606, 0x06FD, 0xFA06,
    0x0603, 0x06FA, 0xFAFD, 0xFA04, 0xFAFC, 0x0604, 0x06FC, 0x04F9, 0xFC07,
    0x0407, 0xFCF9, 0xFA02, 0xFBF9, 0x0507, 0xFAFE, 0x05F9, 0x06FE, 0x0602,
    0xFB07, 0x06F9, 0xFAF9, 0xFA07, 0x0607, 0x07FE, 0xF902, 0x0704, 0xF9FC,
    0xF904, 0xF9FE, 0x07FC, 0x0702, 0x07FA, 0x0706, 0xF906, 0xF9FA, 0x07FD,
    0x0703, 0xF903, 0xF9FD, 0xFB05, 0x0505, 0xFBFB, 0x05FB, 0xFA05, 0x07FB,
    0xF9FB, 0x0605, 0x0705, 0x06FB, 0xFAFB, 0xF905, 0xF9F9, 0xF907, 0x07F9,
    0x0707,
    /* U_2 motion vectors - 442 entries */
    0x0000, 0xFF00, 0x0100, 0x00FF, 0x0001, 0xFFFF, 0x01FF, 0xFF01, 0x0101,
    MV_ESC, 0x0002, 0x00FE, 0xFE00, 0x0200, 0x0003, 0x00FD, 0xFEFF, 0x02FF,
    0xFE01, 0x0201, 0xFF02, 0x0102, 0xFFFE, 0x01FE, 0xFD00, 0x0300, 0x00FC,
    0x0005, 0xFDFF, 0x03FF, 0xFD01, 0x0301, 0xFF03, 0x0103, 0xFFFD, 0x01FD,
    0xFE02, 0x0202, 0xFEFE, 0x02FE, 0xFCFF, 0x04FF, 0xFC01, 0x0401, 0xFBFF,
    0x05FF, 0xFB01, 0x0501, 0x0004, 0x00FB, 0xFD02, 0x0302, 0xFDFE, 0x03FE,
    0xFF05, 0x0105, 0xFFFB, 0x01FB, 0x0006, 0x00FA, 0xFFFC, 0x01FC, 0xFF04,
    0x0104, 0xFE03, 0x0203, 0xFEFD, 0x02FD, 0xFC00, 0x0400, 0xFF06, 0x0106,
    0xFFFA, 0x01FA, 0xFEFC, 0x02FC, 0xFE04, 0x0204, 0xFD03, 0x0303, 0xFDFD,
    0x03FD, 0xFC02, 0x0402, 0xFCFE, 0x04FE, 0xFAFF, 0x06FF, 0xFA01, 0x0601,
    0xFC03, 0x0403, 0xFCFD, 0x04FD, 0xFB02, 0x0502, 0xFBFE, 0x05FE, 0xFB00,
    0x0500, 0xFA00, 0x0600, 0xFE05, 0x0205, 0xFEFB, 0x02FB, 0xFA02, 0x0602,
    0xFAFE, 0x06FE, 0xFD05, 0x0305, 0xFDFB, 0x03FB, 0xFB03, 0x0503, 0xFBFD,
    0x05FD, 0xFC05, 0x0405, 0xFCFB, 0x04FB, 0xFDFC, 0x03FC, 0x00F9, 0xFD04,
    0x0304, 0x0007, 0xF902, 0x0702, 0xFA03, 0x0603, 0xFAFD, 0xFA05, 0xFB05,
    0x06FD, 0x0505, 0x0605, 0xF9FE, 0xFE06, 0x0206, 0xFEFA, 0x07FE, 0x02FA,
    0xFAFB, 0xFBFB, 0x05FB, 0x06FB, 0xFD06, 0x0306, 0xFDFA, 0x03FA, 0x0406,
    0xFC06, 0xFCFA, 0x04FA, 0xFCFC, 0x04FC, 0xFC04, 0x0404, 0xF9FF, 0x07FF,
    0xF901, 0x0701, 0xFFF9, 0x01F9, 0x0107, 0xFF07, 0xFAFC, 0xFBFC, 0x05FC,
    0x06FC, 0xFA04, 0xFB04, 0x0504, 0x0604, 0xFB06, 0x0506, 0xFBFA, 0x05FA,
    0x0008, 0x00F8, 0xF9FC, 0x07FC, 0xF904, 0x0704, 0xFA06, 0x0606, 0xFAFA,
    0x06FA, 0x00F7, 0x0009, 0xF700, 0xF800, 0xF900, 0x0700, 0x0800, 0x0900,
    0xFEF9, 0x02F9, 0xFE07, 0x0207, 0xF8FF, 0xFF08, 0x0108, 0xFFF8, 0x08FF,
    0x01F8, 0xF801, 0x0801, 0x0307, 0xFDF9, 0x03F9, 0xFD07, 0xF7FF, 0x09FF,
    0xF701, 0x0901, 0xF8FC, 0xF903, 0x08FC, 0xF804, 0xF9FD, 0x0804, 0xF905,
    0x0705, 0x0703, 0xF9FB, 0x07FD, 0x07FB, 0xFC07, 0x01F7, 0xFCF9, 0x04F9,
    0x0407, 0xFF09, 0x0109, 0xFFF7, 0x0802, 0xFAF9, 0xFBF9, 0x05F9, 0x06F9,
    0xFB07, 0xF8FE, 0xFA07, 0x0507, 0x08FE, 0x0607, 0xF802, 0x0902, 0xF803,
    0x0803, 0xF8FD, 0x08FD, 0xF805, 0xF7FE, 0x0805, 0x00F6, 0x09FE, 0x000A,
    0xF8FB, 0x08FB, 0xF702, 0x07FA, 0xF906, 0xFEF8, 0x0208, 0xF9FA, 0xF6FF,
    0xFD08, 0x0308, 0xFDF8, 0x0AFF, 0x0706, 0x02F8, 0x03F8, 0xF601, 0xFE08,
    0x0A01, 0xFE09, 0x02F7, 0x0905, 0xF705, 0xF7FB, 0x0209, 0xFEF7, 0x09FB,
    0xF603, 0xFC09, 0xF703, 0x03F7, 0xF605, 0x04F7, 0x0903, 0x0A03, 0x09FD,
    0xFBF7, 0x0AFD, 0x0A05, 0x05F7, 0xFD09, 0x0A00, 0xFC08, 0xF7FD, 0x0408,
    0xFCF8, 0xFB09, 0x0509, 0x0AFB, 0xF600, 0xF6FB, 0xFCF7, 0xF6FD, 0x04F8,
    0x0309, 0xFDF7, 0x0409, 0x01F6, 0xFA09, 0xF9F9, 0xFF0A, 0x08FA, 0xFBF8,
    0xFFF6, 0x07F9, 0x0609, 0xFB08, 0x06F7, 0xF8FA, 0x0906, 0xF907, 0x0508,
    0xF706, 0xF7FA, 0x09FA, 0xFAF7, 0x010A, 0x0707, 0x0806, 0x05F8, 0xF806,
    0x0A06, 0xF7FC, 0xF606, 0xFEF6, 0xF6FE, 0x0608, 0xF908, 0x0A02, 0xF602,
    0x020A, 0xFD0A, 0x09FC, 0x0AFA, 0xFDF6, 0x0708, 0xF9F8, 0x0AFE, 0xFAF8,
    0xF704, 0xFA08, 0x06F8, 0x030A, 0x02F6, 0xF6FA, 0xFE0A, 0x07F8, 0x0904,
    0x03F6, 0x04F6, 0xFBF6, 0x08F9, 0xF8F9, 0x05F6, 0xFB0A, 0x0807, 0xFC0A,
    0xF807, 0x040A, 0x050A, 0xFCF6, 0x0709, 0x060A, 0x07F7, 0xF9F7, 0xF909,
    0xFAF6, 0xFA0A, 0x06F6, 0x0AFC, 0xF808, 0x09F7, 0x08F8, 0xF708, 0x09F9,
    0xF8F7, 0xF707, 0x0809, 0xF7F9, 0xF8F8, 0x09F8, 0xF709, 0xF6FC, 0x08F7,
    0xF809, 0x0907, 0xF604, 0xF90A, 0x0909, 0xF9F6, 0xF7F8, 0x0A04, 0x070A,
    0x07F6, 0x0808, 0x0908, 0xF7F7, 0x080A, 0x08F6, 0xF8F6, 0xF70A, 0x09F6,
    0xF80A, 0x090A, 0xF7F6, 0x0A09, 0x0AF8, 0xF6F6, 0x0AF9, 0xF6F7, 0x0A07,
    0x0A0A, 0x0AF7, 0x0AF6, 0xF60A, 0x0A08, 0xF6F9, 0xF609, 0xF607, 0xF6F8,
    0xF608,
    /* V_1 motion vectors - 226 entries */
    0x0001, 0x00FF, 0x0000, 0xFF00, 0x0100, 0x0002, 0x00FE, 0x00FD, 0x0003,
    0x0004, 0x00FC, 0x00FB, 0x0005, 0x0101, 0xFFFF, 0x01FF, 0xFF01, 0xFE00,
    0x0200, 0xFD00, 0x0300, MV_ESC, 0xFF02, 0x0102, 0xFFFE, 0x01FE, 0xFC00,
    0x0400, 0x0201, 0xFEFF, 0x02FF, 0xFE01, 0xFFFD, 0x01FD, 0xFF03, 0x0103,
    0x00FA, 0x0006, 0xFE02, 0x0202, 0xFEFE, 0x02FE, 0xFD02, 0x0302, 0xFDFE,
    0x03FE, 0xFEFD, 0xFC02, 0x02FD, 0x0402, 0xFE03, 0xFCFE, 0x0203, 0x04FE,
    0xFF04, 0x0104, 0xFFFC, 0x01FC, 0xFB00, 0x0500, 0x00F9, 0x0007, 0xFE04,
    0x0204, 0xFEFC, 0x02FC, 0xFA00, 0x0600, 0xFD04, 0x0304, 0xFDFC, 0x03FC,
    0xFFFB, 0x01FB, 0xFF05, 0x0105, 0x0301, 0xFDFF, 0x03FF, 0xFD01, 0x01F9,
    0x0107, 0xFF07, 0xFFF9, 0x0401, 0xFCFF, 0x04FF, 0xFC01, 0xFEFB, 0x02FB,
    0xFE05, 0x0205, 0xFF06, 0x0106, 0xFFFA, 0x01FA, 0x0501, 0xFBFF, 0x05FF,
    0xFB01, 0x0206, 0xFEFA, 0x02FA, 0xFE06, 0xF900, 0x0700, 0xFB02, 0x0502,
    0xFBFE, 0x05FE, 0xFDFB, 0x0305, 0x03FB, 0xFD05, 0xFDFD, 0x03FD, 0xFD03,
    0x0303, 0x0404, 0xFC04, 0xFCFC, 0x04FC, 0xFA02, 0x0602, 0xFAFE, 0x06FE,
    0xFCFD, 0x04FD, 0x0403, 0xFC03, 0x05FD, 0xFB03, 0x0503, 0xFBFD, 0x02F9,
    0xFE07, 0xFEF9, 0x0207, 0x0601, 0xFAFF, 0xFDFA, 0xFD06, 0x06FF, 0x03FA,
    0xFA01, 0x0306, 0x0504, 0xFB04, 0xFBFC, 0x05FC, 0xFAFC, 0xFA04, 0x0604,
    0x06FC, 0xFA03, 0x0603, 0x06FD, 0xFAFD, 0xF904, 0x0704, 0xF9FC, 0x07FC,
    0x0307, 0xFD07, 0x03F9, 0xFDF9, 0xFCF9, 0xFC07, 0x04F9, 0x0407, 0xFCFB,
    0xFC05, 0x0405, 0x04FB, 0xFBFB, 0xFB05, 0x0505, 0x05FB, 0x04FA, 0xFCFA,
    0x0406, 0xFC06, 0xF901, 0x0701, 0xF9FF, 0x07FF, 0x05F9, 0x0507, 0xFBF9,
    0xFB07, 0x0702, 0xF902, 0xF9FE, 0x07FE, 0x06FB, 0x0605, 0xFAFB, 0xFA05,
    0x0506, 0xF9FD, 0xFB06, 0xFBFA, 0xF903, 0x07FD, 0x05FA, 0x0703, 0xF905,
    0xF9FB, 0x0705, 0x07FB, 0xFA06, 0x0606, 0xFA07, 0x0607, 0xFAF9, 0x06F9,
    0x06FA, 0xFAFA, 0xF906, 0x07FA, 0xF9FA, 0x0706, 0xF907, 0xF9F9, 0x07F9,
    0x0707,
    /* V_2 motion vectors - 442 elements */
    0x0000, 0x00FF, 0x0001, 0xFF00, 0x0100, 0xFFFF, 0x01FF, 0xFF01, 0x0101,
    MV_ESC, 0xFE00, 0x0200, 0xFEFF, 0x02FF, 0xFE01, 0x0201, 0x0002, 0x00FE,
    0x00FC, 0x0003, 0x0004, 0x00FD, 0x0005, 0x00FB, 0xFDFF, 0x03FF, 0xFD01,
    0x0301, 0xFFFC, 0x01FC, 0xFF03, 0x0103, 0xFF04, 0x0104, 0xFFFD, 0x01FD,
    0xFD00, 0x0300, 0xFF02, 0x0102, 0xFFFE, 0x01FE, 0xFC00, 0x0400, 0xFEFC,
    0x02FC, 0xFE03, 0x0203, 0xFE04, 0x0204, 0xFEFD, 0x02FD, 0xFCFF, 0x04FF,
    0xFC01, 0x0401, 0xFE02, 0x0202, 0xFEFE, 0x02FE, 0xFD02, 0x0302, 0xFDFE,
    0x03FE, 0xFDFC, 0x03FC, 0xFD03, 0x0303, 0xFD04, 0x0304, 0xFDFD, 0x03FD,
    0xFF05, 0x0105, 0xFFFB, 0x01FB, 0x0006, 0x00FA, 0xFBFF, 0x05FF, 0xFB01,
    0x0501, 0xFAFF, 0x06FF, 0xFA01, 0x0601, 0x000A, 0xFF06, 0x0106, 0xFFFA,
    0x01FA, 0xFE05, 0x0205, 0xFB00, 0x0500, 0xFEFB, 0x02FB, 0xFE06, 0x0206,
    0xFEFA, 0x02FA, 0xFC02, 0x0402, 0xFCFE, 0x04FE, 0xFA00, 0x0600, 0xFCFC,
    0xFC03, 0x04FC, 0x0403, 0xFC04, 0xFCFD, 0x0404, 0x04FD, 0xFB02, 0x0502,
    0xFBFE, 0x05FE, 0xFD05, 0x0305, 0xFDFB, 0x03FB, 0x00F7, 0x00F9, 0x00F6,
    0x0007, 0x0009, 0xFA02, 0xFBFC, 0x0602, 0xFB03, 0x05FC, 0x0503, 0xFB04,
    0xFBFD, 0x0504, 0x05FD, 0xFAFE, 0x06FE, 0xF9FF, 0x07FF, 0xF901, 0x0701,
    0xFC05, 0x0405, 0xFCFB, 0x04FB, 0xFAFC, 0xFA03, 0x06FC, 0x0603, 0xFFF9,
    0xFA04, 0x01F9, 0xFAFD, 0x0604, 0xFB05, 0x06FD, 0x0505, 0xFF07, 0x0107,
    0xFBFB, 0x05FB, 0xFC06, 0xFD06, 0x0306, 0x0406, 0xFCFA, 0xFDFA, 0x03FA,
    0x04FA, 0xF9FC, 0x01F7, 0xF903, 0x07FC, 0x0703, 0xF904, 0xF9FD, 0x0704,
    0x07FD, 0xFB06, 0x0506, 0xFBFA, 0x05FA, 0xFF09, 0x0109, 0xFFF7, 0xF900,
    0x0700, 0xF8FF, 0x08FF, 0xF801, 0x0801, 0xFEF9, 0x02F9, 0xFA06, 0x0606,
    0xFAFA, 0xFE07, 0x06FA, 0xF800, 0x0800, 0x0207, 0xF8FC, 0xF803, 0x08FC,
    0x0803, 0xF804, 0xF8FD, 0x0804, 0x08FD, 0xFF08, 0x0108, 0xFFF8, 0xF700,
    0x00F8, 0x01F8, 0x0900, 0x0008, 0xF902, 0x0702, 0xF9FE, 0x07FE, 0xFDF9,
    0x03F9, 0x0307, 0xFD07, 0xF7FF, 0x09FF, 0xF701, 0x0901, 0xFA05, 0x0605,
    0xFAFB, 0x06FB, 0xFBF9, 0xFCF9, 0x04F9, 0x05F9, 0x0507, 0xFB07, 0x0407,
    0xFC07, 0xFE09, 0x02F7, 0x0209, 0xF600, 0x0A00, 0xFEF7, 0x0802, 0x0705,
    0xF905, 0xFFF6, 0xF8FE, 0x08FE, 0x01F6, 0x010A, 0xF9FB, 0xFF0A, 0x07FB,
    0xF802, 0x03F7, 0x0309, 0xFD09, 0xFDF7, 0x0607, 0x0902, 0xFAF9, 0x0409,
    0x04F7, 0x06F9, 0xF7FE, 0x09FE, 0xFA07, 0xFC09, 0xFCF7, 0xF702, 0xF705,
    0x07FA, 0xF9F9, 0x0707, 0xF7FB, 0x07F9, 0x0805, 0x0A05, 0xF805, 0xF906,
    0x0706, 0xF9FA, 0xF907, 0xF605, 0x0905, 0xF6FB, 0xF8FB, 0x08FB, 0x09FB,
    0x0AFB, 0xFE08, 0x0509, 0xFEF6, 0x020A, 0xF6FF, 0xFD08, 0x0308, 0xFDF8,
    0xFB09, 0x0208, 0x0AFF, 0xFE0A, 0x02F8, 0xFBF7, 0x02F6, 0x03F8, 0xFEF8,
    0xF601, 0x05F7, 0x0A01, 0xFB08, 0xF7FC, 0x0904, 0xF703, 0x0A04, 0x09FC,
    0x0AFC, 0x0903, 0xF604, 0x0608, 0x09FD, 0xFCF8, 0xFAF8, 0xF7FD, 0x05F8,
    0xFA08, 0x0508, 0xF704, 0xF603, 0xFC08, 0x0A03, 0xF6FC, 0x04F8, 0xFBF8,
    0x06F8, 0xF6FD, 0x0408, 0x0AFD, 0xF706, 0x08F9, 0xF908, 0xF8F9, 0x040A,
    0xF602, 0x0A02, 0x0708, 0x08FA, 0xFDF6, 0x0AFE, 0xF806, 0xFCF6, 0xF7FA,
    0x0906, 0xFC0A, 0x0807, 0x07F8, 0xF6FE, 0x03F6, 0x030A, 0x09FA, 0xF8FA,
    0xF9F8, 0xFD0A, 0x0806, 0xF807, 0x04F6, 0xF7F9, 0xF707, 0xFBF6, 0x09F9,
    0x05F6, 0xFB0A, 0x0907, 0x050A, 0xF6F9, 0x0AF9, 0x0A07, 0xF607, 0xF909,
    0xF8F6, 0xF70A, 0x09F6, 0xFA09, 0x07F7, 0x0609, 0x070A, 0xFAF7, 0x06F7,
    0xF90A, 0xFA0A, 0x090A, 0xFAF6, 0xF9F6, 0xF7F6, 0x080A, 0xF80A, 0x06F6,
    0xF9F7, 0x07F6, 0x0709, 0x060A, 0x08F6, 0x0909, 0x09F8, 0xF7F7, 0x08F8,
    0xF8F8, 0xF709, 0x0809, 0x0808, 0xF808, 0x0908, 0xF7F8, 0xF8F7, 0x08F7,
    0xF708, 0x09F7, 0xF809, 0x0AFA, 0xF60A, 0x0A06, 0x0AF6, 0xF609, 0xF6F7,
    0xF6FA, 0x0A09, 0x0AF7, 0xF6F8, 0x0A0A, 0x0AF8, 0xF606, 0x0A08, 0xF608,
    0xF6F6,
};

static const uint8_t clv_bias_len_counts[][16] = {
    {  1,  1,  1,  0,  2,  0,  4,  3,  3,  6,  5, 10, 16,  7, 17,  2 },
    {  1,  0,  2,  2,  0,  4,  2,  6,  4,  8,  6, 10,  9,  7, 23, 14 },
    {  0,  2,  1,  3,  2,  3,  4,  4,  8,  8,  8,  7,  7, 10, 21,  6 },
    {  1,  1,  1,  0,  2,  1,  3,  2,  4,  4,  2,  8,  6,  2,  3,  2 },
    {  1,  0,  2,  2,  1,  3,  2,  4,  4,  4,  4,  4,  2,  7,  6,  8 },
    {  1,  1,  1,  0,  2,  2,  1,  3,  2,  4,  3,  5,  6,  6,  3,  2 },
    {  1,  0,  2,  2,  2,  1,  3,  2,  4,  5,  1,  4, 10,  2,  3,  2 },
};

#define BIAS_ESC 1
static const uint16_t clv_bias_syms[] = {
    /* Y_1 bias values - 78 entries */
    0x0000, 0xFFFC, 0x0004, 0xFFF8, 0x0008, 0xFFF4, 0x000C, 0xFFF0, 0x0010,
    0xFFEC, 0x0014, 0xFFE8, 0x0018, 0xFFE4, 0x001C, 0x0020, 0xFFE0, 0x0024,
    0xFFDC, 0x0028, 0xFFD8, 0x002C, 0xFFD4, 0x0030, 0xFFD0, 0xFFCC, 0x0034,
    0xFFC8, 0x0038, BIAS_ESC, 0xFFC4, 0x003C, 0xFFC0, 0x0040, 0xFFBC, 0x0044,
    0xFFB8, 0x0048, 0xFFB4, 0x004C, 0xFFB0, 0x0050, 0xFFAC, 0x0054, 0xFFA8,
    0x0058, 0xFFA4, 0x005C, 0x0060, 0xFFA0, 0x0064, 0xFF9C, 0x0068, 0xFF98,
    0x006C, 0xFF94, 0x0070, 0xFF90, 0x0074, 0xFF8C, 0xFF88, 0x0078, 0x007C,
    0xFF84, 0xFF80, 0x0080, 0x0088, 0xFF78, 0xFF7C, 0x0084, 0x008C, 0xFF74,
    0x0090, 0xFF70, 0x0094, 0xFF6C, 0xFF68, 0x0098,
    /* Y_2 bias values - 98 entries */
    0x0000, 0xFFFC, 0x0004, 0xFFF8, 0x0008, 0xFFF4, 0x000C, 0xFFF0, 0x0010,
    0xFFEC, 0x0014, 0xFFE8, 0x0018, 0xFFE4, 0x001C, 0xFFE0, 0x0020, 0xFFDC,
    0x0024, 0xFFD8, 0x0028, 0xFFD4, 0x002C, 0xFFD0, 0x0030, 0xFFCC, 0x0034,
    0xFFC8, 0x0038, 0x003C, 0xFFC4, 0x0040, 0xFFC0, 0xFFBC, 0x0044, 0x0048,
    0xFFB8, 0x004C, 0xFFB4, 0x0050, 0xFFB0, 0x0054, 0xFFAC, 0xFFA8, 0x0058,
    0x005C, 0xFFA4, 0x0060, 0xFFA0, 0x0064, 0xFF9C, 0xFF98, 0x0068, 0x006C,
    0xFF94, 0x0070, 0xFF90, 0x0074, 0xFF8C, 0xFF88, 0x0078, 0xFF84, 0x007C,
    0xFF80, 0x0080, 0xFF7C, 0x0084, BIAS_ESC, 0xFF78, 0x0088, 0x008C, 0xFF74,
    0x0090, 0xFF70, 0x0094, 0xFF6C, 0xFF64, 0x009C, 0xFF68, 0x0098, 0xFF60,
    0x00A0, 0xFF5C, 0x00A4, 0x00A8, 0xFF58, 0x00AC, 0xFF54, 0xFF50, 0x00B0,
    0x00B4, 0xFF4C, 0xFF48, 0x00B8, 0xFF44, 0x00BC, 0xFF40, 0x00C0,
    /* Y_3 bias values - 94 entries */
    0x0000, 0xFFFC, 0x0004, 0xFFF8, 0x0008, 0xFFF4, 0x000C, 0xFFF0, 0x0010,
    0xFFEC, 0x0014, 0xFFE8, 0x0018, 0xFFE4, 0x001C, 0xFFE0, 0x0020, 0xFFDC,
    0x0024, 0xFFD8, 0x0028, 0xFFD4, 0x002C, 0xFFD0, 0x0030, 0xFFCC, 0x0034,
    0xFFC8, 0x0038, 0xFFC4, 0x003C, 0xFFC0, 0x0040, 0x0044, 0xFFBC, 0x0048,
    0xFFB8, 0x004C, 0xFFB4, 0x0050, 0xFFB0, 0x0054, 0xFFAC, 0x0058, 0xFFA8,
    0x005C, 0xFFA4, 0xFFA0, 0x0060, 0x0064, 0xFF9C, 0x0068, 0xFF98, 0x006C,
    0xFF94, 0xFF90, 0x0070, 0xFF8C, 0x0074, 0x0078, 0xFF88, 0xFF84, 0x007C,
    0xFF80, 0x0080, 0x0084, 0xFF7C, 0x0088, 0xFF78, 0x008C, 0xFF74, 0xFF70,
    0x0090, 0x0094, 0xFF6C, 0x0098, 0xFF68, 0xFF64, 0x009C, 0xFF60, 0x00A0,
    0xFF5C, 0x00A4, BIAS_ESC, 0xFF58, 0x00A8, 0x00AC, 0xFF54, 0xFF50, 0x00B0,
    0xFF4C, 0x00B4, 0x00B8, 0xFF48,
    /* U_1 bias values - 42 entries */
    0x0000, 0xFFFC, 0x0004, 0xFFF8, 0x0008, 0xFFF4, 0x000C, 0xFFF0, 0x0010,
    0xFFEC, 0x0014, 0x0018, 0xFFE8, 0xFFE4, 0x001C, 0xFFE0, 0x0020, 0xFFDC,
    0x0024, 0xFFD8, 0x0028, 0x002C, 0xFFD4, 0x0030, 0xFFCC, 0xFFD0, 0x0034,
    0xFFC8, 0x0038, 0x003C, 0xFFC4, 0xFFC0, 0x0040, 0xFFBC, 0x0044, 0xFFB8,
    0x0048, 0xFFB4, 0x004C, 0x0050, 0xFFB0, BIAS_ESC,
    /* U_2 bias values - 54 entries */
    0x0000, 0xFFFC, 0x0004, 0xFFF8, 0x0008, 0xFFF4, 0x000C, 0xFFF0, 0x0010,
    0x0014, 0xFFEC, 0x0018, 0xFFE8, 0x001C, 0xFFE4, 0x0020, 0xFFE0, 0xFFDC,
    0x0024, 0xFFD8, 0x0028, 0xFFD4, 0x002C, 0x0030, 0xFFD0, 0x0034, 0xFFCC,
    0x0038, 0xFFC8, 0x003C, 0xFFC4, 0x0040, 0xFFC0, 0xFFBC, 0x0044, 0xFFB4,
    0xFFB8, 0x0048, 0x004C, BIAS_ESC, 0x0058, 0xFFB0, 0xFFA8, 0x0054, 0xFFAC,
    0x0050, 0xFF9C, 0x005C, 0xFFA0, 0x0068, 0xFF98, 0xFFA4, 0x0064, 0x0060,
    /* V_1 bias values - 42 entries */
    0x0000, 0xFFFC, 0x0004, 0xFFF8, 0x0008, 0xFFF4, 0x000C, 0xFFF0, 0x0010,
    0xFFEC, 0x0014, 0x0018, 0xFFE8, 0xFFE4, 0x001C, 0xFFE0, 0x0020, 0xFFDC,
    0x0024, 0xFFD8, 0x0028, 0x002C, 0xFFD4, 0x0030, 0xFFD0, 0xFFCC, 0x0034,
    0xFFC8, 0x0038, 0x003C, 0xFFC4, BIAS_ESC, 0xFFC0, 0x0040, 0xFFBC, 0x0044,
    0xFFB8, 0x0048, 0xFFB4, 0x004C, 0x0050, 0xFFB0,
    /* V_2 bias values - 44 entries */
    0x0000, 0xFFFC, 0x0004, 0xFFF8, 0x0008, 0xFFF4, 0x000C, 0xFFF0, 0x0010,
    0xFFEC, 0x0014, 0xFFE8, 0x0018, 0x001C, 0xFFE4, 0xFFE0, 0x0020, 0xFFDC,
    0x0024, 0x0028, 0xFFD8, 0x002C, 0xFFD4, 0x0030, 0xFFD0, 0xFFCC, 0x0034,
    0xFFC8, 0x0038, 0x0040, 0xFFC4, 0xFFC0, 0x003C, 0x0044, 0xFFBC, 0xFFB8,
    0x0048, 0xFFB4, 0x004C, BIAS_ESC, 0xFFB0, 0x0054, 0xFFAC, 0x0050
};

#endif /* AVCODEC_CLEARVIDEODATA_H */
